#include <c167.h>

#include "cc2.h"


//--- defines -------------------------------------------------------


//--- Type defs -----------------------------------------------------


//--- Variables -----------------------------------------------------


//--- Functions -----------------------------------------------------


/*-------------------------------------------------------------------------*

    Name    : Init

    Function: Inits the i/o routines.

    Input   : (none)

    Returns : (none)

 *-------------------------------------------------------------------------*/

void Cc2Init( void )
{
	// Disable all interrupts.
	CLR_SFRBIT( IEN );

	// i2c outputs
	SET_SFRBIT( dp3.8 );	// SDA
	SET_SFRBIT( dp3.9 );	// SCL

	// Reset RTC
	RTCL = 0;
	RTCH = 0;

    /* Init the pwm outputs 0 and 1.
    */
	DP8  = 0x0007;		// make 3 pwm outputs
	ODP8 = 0x0000;		// push/pull outputs
	CCM4 = ( CCM4 & 0xff00 ) | 0x0077;

	/*	Init pwm output 3 for beeper.
	*/
	// Set timer 8: run and 400 ns
	T78CON = ( T78CON & 0x00ff ) | 0x4000;
	T8REL = 0xff01;
	CCM4 = ( CCM4 & 0xf0ff) | 0x0f00;

} // Cc2Init


int ReadAdc( int channel )
{
	ADCON = channel | 0x0080;
	WAIT_UNTIL_BIT_CLR( adcon.8 );

	return( ADDAT & 0x03ff );

} // ReadAdc


void InitPwm( int channel, int timebase, int length )
{
	// Set timer 7
	T78CON = ( T78CON & 0xff00 ) | ( ( timebase % 8 ) | 0x0040 );
	T7REL = -length;

} // InitPwm


void SetPwm( int channel, int value )
{
	if ( channel == 0 )
	{
		CC16 = -value;
	}
	else
	{
		CC17 = -value;
	}

} // SetPwm


void Beep( int tone )
{
	if ( tone >= 0 )
	{
		/*	timebase is 400 ns
			tone = 12 => freq = 440 Hz => time = 2300 us => 5750 tics of 400 ns
			so 5750 / 12 = 479 tics per tone
		*/
		T8REL = -(tone % 40) * 479;

		/*	Compare value lies halfway between reload value and 0xffff.
		*/
		CC18  = (int)T8REL / 2;
	}
	else
	{
		CC18 = 0;
	}

} // Beep


void Sleep( int time )
{
} // Sleep


long Timer( void )
{
	unsigned int msw1;
	unsigned int msw2;
	unsigned int lsw;
	union 
	{
		struct
		{
			int lsw;
			int msw;
		} ints;
		long l;
	} result;

	msw1 = RTCH;
	lsw  = RTCL;
	msw2 = RTCH;

	/*	Handle overflow during read actions.
	*/
	if ( ( msw1 != msw2 ) && ( lsw > 0x8000 ) )
	{
		result.ints.msw = msw2;
	}
	else
	{
		result.ints.msw = msw1;
	}
	result.ints.lsw = lsw;

	return ( result.l );

} // Timer



void Delay( int i )
{
	while ( i > 0 )
	{
		i--;
	}

} // Delay


void I2cStart( void )
{
	SET_SDA;
	SET_SCL;
	CLR_SDA;
	Delay( 100 );
	CLR_SCL;

} // I2cStart


void I2cStop( void )
{
	CLR_SDA;
	SET_SCL;
	Delay( 100 );
	SET_SDA;

} // I2cStop


void I2cSend( int byte )
{
	int i;

	/*	Send 8 data bits.
		msb first
	*/
	for ( i = 7; i >= 0; i-- )
	{
		CLR_SCL;
		Delay( 100 );

		if ( ( byte & ( 1 << i ) ) != 0 )
		{
			SET_SDA;
		}
		else
		{
			CLR_SDA;
		}
		SET_SCL;
		Delay( 100 );
	}

	// Request acknowlegde.
	CLR_SCL;
	CLR_SFRBIT( dp3.9 );	// SDA as input
	Delay( 100 );
	SET_SCL;
	Delay( 100 );
	CLR_SCL;
	Delay( 100 );
	SET_SFRBIT( dp3.9 );	// SDA as output
	CLR_SDA;

} // I2cSend


void LcdClear( void )
{
	I2cStart();
	I2cSend( 0x74 );	// i2c address of internal display
	I2cSend( 0x00 );
	I2cSend( 0x01 );
	I2cSend( 0x84 );
	I2cStop();

} // LcdClear


void LcdGoto( int line, int pos )
{
	I2cStart();
	I2cSend( 0x74 );	// i2c address of internal display
	I2cSend( 0x00 );
	I2cSend( 0x84 + (line << 6) + pos );
	I2cStop();

} // LcdGoto


void LcdPut( char * sz )
{
	I2cStart();
	I2cSend( 0x74 );	// i2c address of internal display
	I2cSend( 0x40 );
	while ( *sz != 0 )
	{
		I2cSend( *(sz++) + 128 );
	}
	I2cStop();

} // LcdPut


void SerInit( int speed )
{
	switch ( speed )
	{
		case SPEED_300:
		{
			S0BG = 0x0822;
			break;
		}
		case SPEED_600:
		{
			S0BG = 0x0410;
			break;
		}
		case SPEED_1200:
		{
			S0BG = 0x0208;
			break;
		}
		case SPEED_2400:
		{
			S0BG = 0x0104;
			break;
		}
		case SPEED_4800:
		{
			S0BG = 0x0081;
			break;
		}
		default:
		case SPEED_9600:
		{
			S0BG = 0x0040;
			break;
		}
		case SPEED_19200:
		{
			S0BG = 0x001f;
			break;
		}
		case SPEED_38400:
		{
			S0BG = 0x000f;
			break;
		}
		case SPEED_57600:
		{
			S0BG = 0x000a;
			break;
		}
		case SPEED_115200:
		{
			S0BG = 0x0004;
			break;
		}
	}

} // SerInit


void SerPut( char ch )
{
	S0TBUF = ch;

	// Wait until transmission ready.
	while ( ( S0TIC & 0x0080) == 0 );
	S0TIC &= 0xff7f;

} // SerPut


void SerSend( char * sz )
{
	while ( *sz != 0 )
	{
		SerPut( *(sz++) );
	}

} // SerOut


// --- End Of File --------------------------------------------------
